﻿using System;

using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;
using MettlerToledo.IND780.SharedDataClient;
using MettlerToledo.IND780.Softkeys;

namespace Classify_Application
{
	public partial class Form4 : Form
	{
		private SDclient SD_client = ClassifyMain.SD_client;

		private Softkeys mySoftkeys;

		private bool CtrlKeyPressed = false;

		private ComboBox[] cbArray = new ComboBox[6];

		// shared data to store comboBox selections
		private string[] sdArray = {"az0201",		// com port name
									"ax0202",		// baud rate
									"ax0203",		// data bits
								    "ax0204",		// parity
									"ax0205",		// stop bits
									"ax0206"};		// flow control

		//public static SerialPort serialPort_1 = new SerialPort();
		//public static SerialPort serialPort_2 = new SerialPort();

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Form4
		* Description   :Constructor
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		public Form4()
		{
			InitializeComponent();

			if (SD_client.GetSharedDataConnectionStatus)
				this.label_connected.BackColor = Color.Lime;

			this.SoftkeysInitialization();

			this.RegisterSoftkeys();

			this.Load_ComboBoxSelections();

			this.comboBox_com_port.Focus();
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Load_ComboBoxSelections
		* Description   :This method reads each shared data assigned to the comboBoxes and
		*				:sets their selected index values.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void Load_ComboBoxSelections()
		{
			ComboBox[] cbxArray = {this.comboBox_com_port, this.comboBox_baud, this.comboBox_data_bits,
								  this.comboBox_parity, this.comboBox_stop_bits, this.comboBox_flow_control};

			for (int i = 0; i < cbxArray.Length; i++)
			{
				int sdValue = 0;
				string sdValueString = string.Empty;

				this.cbArray[i] = cbxArray[i];

				// load the shared data values
				sdValueString = SD_client.ReadSharedData(this.sdArray[i]);

				if (i == 0)		// com port name
				{
					if (sdValueString == "COM1")
						this.cbArray[i].SelectedIndex = 0;
					else if (sdValueString == "COM2")
						this.cbArray[i].SelectedIndex = 1;
					else
						continue;
				}
				else
				{
					try
					{
						sdValue = int.Parse(sdValueString);
					}
					catch
					{
						sdValue = 0;
					}
					try
					{

						this.cbArray[i].SelectedIndex = sdValue;
					}
					catch
					{
						this.cbArray[i].SelectedIndex = 0;
					}
				}
			}
		}


		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::SoftkeysInitialization
		* Description   :Creates the five picture boxes and draws the softkey borders used by the softkeys
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void SoftkeysInitialization()
		{
			mySoftkeys = new Softkeys();
			mySoftkeys.AllocateSoftkeys(this, 175, @"Storage Card\Terminal\SKBMP\COLOR\");
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::RegisterSoftkeys
		* Description   :Registers the softkey graphic and method used by each softkey.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void RegisterSoftkeys()
		{
			// level 0 softkeys
			mySoftkeys.LoadSoftkeys(0, 0, true, @"Storage Card\Terminal\SKBMP\COLOR\exit.bmp", (Softkeys.mySKMethodDelegate)ComConfigureExit);
			mySoftkeys.LoadSoftkeys(0, 1, true, @"Storage Card\Terminal\SKBMP\COLOR\connect.bmp", (Softkeys.mySKMethodDelegate)ComConnect);
			mySoftkeys.LoadSoftkeys(0, 2, true, @"Storage Card\Terminal\SKBMP\COLOR\disconnect.bmp", (Softkeys.mySKMethodDelegate)ComDisconnect);

			mySoftkeys.NumberOfLevels = 1;

			mySoftkeys.DisplaySoftKeys(0);
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::ComConnect
		* Description   :Connects the serial port.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ComConnect()
		{
			int[] baudRates = { 300, 600, 1200, 2400, 4800, 9600, 19200, 38400, 57600, 115200, 999999 };
			Parity[] parity = { Parity.None, Parity.Even, Parity.Odd, Parity.Mark, Parity.Space };
			StopBits[] stopbits = { StopBits.One, StopBits.OnePointFive, StopBits.Two };

			string portName = "";
			SerialPort portBeingConfigured = null;

			try
			{
				portName = ClassifyMain.SD_client.ReadSharedData("az0201");

				if (portName == "COM1")
					portBeingConfigured = ClassifyMain.serialPort_1;
				else if (portName == "COM2")
					portBeingConfigured = ClassifyMain.serialPort_2;
				else
				{
					this.label_connection_status.Text = "Invalid Port Name";
					this.label_connection_status.ForeColor = Color.Red;
					return;
				}

				if (portBeingConfigured.IsOpen)
					portBeingConfigured.Close();

				if (portName == "COM1")
					portBeingConfigured.PortName = "COM1:";
				else
					portBeingConfigured.PortName = "COM2:";

				portBeingConfigured.BaudRate = baudRates[int.Parse(ClassifyMain.SD_client.ReadSharedData("ax0202"))];
				portBeingConfigured.DataBits = (ClassifyMain.SD_client.ReadSharedData("ax0203").TrimEnd() == "0") ? 7 : 8;
				portBeingConfigured.Parity = parity[int.Parse(ClassifyMain.SD_client.ReadSharedData("ax0204"))];
				portBeingConfigured.StopBits = stopbits[int.Parse(ClassifyMain.SD_client.ReadSharedData("ax0205"))];
				portBeingConfigured.Handshake = (ClassifyMain.SD_client.ReadSharedData("ax0206").TrimEnd() == "0") ? Handshake.None : Handshake.XOnXOff;
				portBeingConfigured.ReadTimeout = 10000;
				portBeingConfigured.WriteTimeout = 500;

				try
				{
					portBeingConfigured.Open();
				}
				catch
				{
					this.label_connection_status.Text = "Serial Port Error";
					this.label_connection_status.ForeColor = Color.Red;
					return;
				}
			}
			catch
			{
				this.label_connection_status.Text = "Serial Port Error";
				this.label_connection_status.ForeColor = Color.Red;
				return;
			}
			this.label_connection_status.Text = "Serial Port Open";
			this.label_connection_status.ForeColor = Color.Lime;
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::ComDisconnect
		* Description   :Disconnects the serial port.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ComDisconnect()
		{
			string portName = ClassifyMain.SD_client.ReadSharedData("az0201");
			SerialPort portToDisconnect = null;

			if (portName == "COM1")
				portToDisconnect = ClassifyMain.serialPort_1;
			else if (portName == "COM2")
				portToDisconnect = ClassifyMain.serialPort_2;
			else
			{
				this.label_connection_status.Text = "Invalid Port Name";
				this.label_connection_status.ForeColor = Color.Red;
				return;
			}

			if (portToDisconnect.IsOpen)
			{
				portToDisconnect.Close();
				this.label_connection_status.Text = "Serial Port Closed";
				this.label_connection_status.ForeColor = Color.Blue;
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::ComConfigureExit
		* Description   :Disconnect from the shared data server and closes the form.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ComConfigureExit()
		{
			this.Close();
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Common_KeyDown
		* Description   :Common key-down event handler for all keys.  Forces a key click noise.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Common_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if (e.KeyCode == Keys.ControlKey)
			{
				this.CtrlKeyPressed = true;
				return;
			}
			ClassifyMain.SD_client.SetSharedData("xc0134", "1");	// key click

			if (this.CtrlKeyPressed)
				e.Handled = true;			// this prevents controls such as comboBoxes from getting the key
		}

#if USE_OLD_COMMON_KEYUP
		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Common_KeyUp
		* Description   :Common key-up event handler for all keys.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Common_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if (e.KeyCode == Keys.ControlKey)
				this.CtrlKeyPressed = false;

			// the enter key will move the focus and scroll bar if necessary
			if (e.KeyCode == Keys.Enter && sender is ComboBox)
			{
				for (int i = 0; i < this.cbArray.Length; i++)
				{
					if (this.cbArray[i].Focused)
					{
						i++;
						if (i == this.cbArray.Length)
							i = 0;
						this.cbArray[i].Focus();
						break;
					}
				}
			}
			else
			{
				if (!mySoftkeys.SoftkeyDispatcher((int)e.KeyCode))
				{
				}
			}
		}
#endif

		// special native code call to force the combobox to drop-down
		// when it receives the focus
		[System.Runtime.InteropServices.DllImport("coredll.dll")]
		public static extern int SendMessage(int hWnd, int msg, int wParam, IntPtr lParam);
		public const int CB_SHOWDROPDOWN = 0x14F;

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Common_KeyUp
		* Description   :Common key-up event handler for all keys.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Common_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if (e.KeyCode == Keys.ControlKey)
				this.CtrlKeyPressed = false;

			if (sender is ComboBox)
			{
				if (e.KeyCode == Keys.Enter)
				{
					for (int i = 0; i < this.cbArray.Length; i++)
					{
						if (this.cbArray[i].Focused)
						{
							i++;
							if (i == this.cbArray.Length)
								i = 0;
							this.cbArray[i].Focus();
							break;
						}
					}
				}
				else if (e.KeyCode == Keys.Right)	// use right arrow to force a dropdown
				{
					SendMessage(((ComboBox)sender).Handle.ToInt32(), CB_SHOWDROPDOWN, 1, IntPtr.Zero);
					e.Handled = true;
				}
				else if (e.KeyCode == Keys.Left)	// use left arrow to retract the dropdown
				{
					SendMessage(((ComboBox)sender).Handle.ToInt32(), CB_SHOWDROPDOWN, 0, IntPtr.Zero);
					e.Handled = true;
				}
				else
				{
					mySoftkeys.SoftkeyDispatcher((int)e.KeyCode);
				}
			}
			else
			{
				mySoftkeys.SoftkeyDispatcher((int)e.KeyCode);
			}
		}


		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::Form4_Paint
		* Description   :The Form1 Paint event handler.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Form4_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			mySoftkeys.DrawSoftkeyBorders(this);
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form4::common_combobox_LostFocus
		* Description   :Common combobox lost focus handler.  Saves selected index number in shared data
		*				:defined in sdArray. 
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void common_combobox_LostFocus(object sender, System.EventArgs e)
		{
			for (int i = 0; i < this.cbArray.Length; i++ )
			{
				if ( (ComboBox)sender == this.cbArray[i] )
				{
					if (i == 0)	// Port name
						ClassifyMain.SD_client.SetSharedData(this.sdArray[i], ((ComboBox)sender).Text);
					else
						ClassifyMain.SD_client.SetSharedData(this.sdArray[i], ((ComboBox)sender).SelectedIndex.ToString());
				}
			}
		}


	}
}